"use client";
import React, { useState, useEffect } from "react";
import { useSearchParams } from "next/navigation";
import Link from "next/link";
import Loading from "@/app/components/Loading";
import axiosInstance from "@/api/axiosInstance";

const SearchPageContent = () => {
  const searchParams = useSearchParams();
  const query = searchParams.get("q") || "";

  const [results, setResults] = useState([]);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    if (query) {
      performSearch(query);
    }
  }, [query]);

  const performSearch = async (searchTerm) => {
    if (!searchTerm.trim()) return;

    try {
      setLoading(true);
      const response = await axiosInstance.get("/search", {
        params: { query: searchTerm.trim(), type: "all" },
      });
      setResults(response.data.results);
    } catch (error) {
      setResults([]);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="font-amiri min-h-screen pt-40 pb-20">
      <div className="container mx-auto lg:px-12 md:px-8 px-5">
        {loading ? (
          <Loading />
        ) : query ? (
          <div className="max-w-4xl mx-auto ">
            <div className="mb-8 text-center">
              <h1 className="text-2xl font-bold text-quaternary mb-2">
                نتائج البحث عن: "{query}"
              </h1>
              <p className="text-gray-600">
                {results.length > 0
                  ? `تم العثور على ${results.length} نتيجة`
                  : "لم يتم العثور على نتائج"}
              </p>
            </div>

            {results.length > 0 ? (
              <div className="space-y-6">
                {results.map((item) => (
                  <Link key={`${item.type}-${item.id}`} href={item.url}>
                    <div className="bg-white mt-3 rounded-xl shadow-md hover:shadow-lg transition p-6 border-r-4 border-quaternary cursor-pointer">
                      <div className="flex justify-between items-start mb-3">
                        <h3 className="text-xl font-bold text-gray-800 flex-1 pr-4">
                          {item.title}
                        </h3>
                        <span
                          className={`px-3 py-1 rounded-full text-sm font-medium ${
                            item.type === "question"
                              ? "bg-blue-100 text-blue-800"
                              : "bg-green-100 text-green-800"
                          }`}
                        >
                          {item.type === "question" ? "سؤال" : "درس"}
                        </span>
                      </div>

                      <div className="flex items-center text-sm text-gray-600 mb-3">
                        <span className="bg-gray-100 px-3 py-1 rounded-full">
                          {item.category}
                        </span>
                        <span className="mx-2">•</span>
                        <span>
                          {new Date(item.created_at).toLocaleDateString("ar-EG")}
                        </span>
                      </div>

                      <p className="text-gray-700 leading-7 line-clamp-2">
                        {item.content}
                      </p>

                      <div className="mt-4 text-quaternary font-medium hover:text-secondary transition">
                        {item.type === "question" ? "عرض السؤال →" : "عرض الدرس →"}
                      </div>
                    </div>
                  </Link>
                ))}
              </div>
            ) : (
              <div className="text-center py-12">
                <div className="text-gray-500 text-lg mb-4">
                  لم نعثر على أي نتائج تطابق "{query}"
                </div>
              </div>
            )}
          </div>
        ) : (
          <div className="text-center py-20">
            <div className="text-gray-500 text-xl mb-4">
              ابحث عن الأسئلة والدروس
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default SearchPageContent;
